import { useForm } from '@inertiajs/react';
import { FormEventHandler, useRef } from 'react';

import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';

import { Dialog, DialogClose, DialogContent, DialogDescription, DialogFooter, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import LoadingButton from './loading-button';
import { Card } from './ui/card';

export default function DeleteUser() {
   const passwordInput = useRef<HTMLInputElement>(null);
   const { data, setData, delete: destroy, processing, reset, errors, clearErrors } = useForm<Required<{ password: string }>>({ password: '' });

   const deleteUser: FormEventHandler = (e) => {
      e.preventDefault();

      destroy(route('profile.destroy'), {
         preserveScroll: true,
         onSuccess: () => closeModal(),
         onError: () => passwordInput.current?.focus(),
         onFinish: () => reset(),
      });
   };

   const closeModal = () => {
      clearErrors();
      reset();
   };

   return (
      <Card className="space-y-6 p-6">
         <header>
            <h3 className="mb-0.5 text-base font-medium">Delete account</h3>
            <p className="text-muted-foreground text-sm">Delete your account and all of its resources</p>
         </header>

         <div className="space-y-4 rounded-lg border border-red-100 bg-red-50 p-4 dark:border-red-200/10 dark:bg-red-700/10">
            <div className="relative space-y-0.5 text-red-600 dark:text-red-100">
               <p className="font-medium">Warning</p>
               <p className="text-sm">Please proceed with caution, this cannot be undone.</p>
            </div>

            <Dialog>
               <DialogTrigger asChild>
                  <Button variant="destructive" className="text-primary-foreground">
                     Delete account
                  </Button>
               </DialogTrigger>

               <DialogContent>
                  <DialogTitle>Are you sure you want to delete your account?</DialogTitle>
                  <DialogDescription>
                     Once your account is deleted, all of its resources and data will also be permanently deleted. Please enter your password to
                     confirm you would like to permanently delete your account.
                  </DialogDescription>
                  <form className="space-y-6" onSubmit={deleteUser}>
                     <div className="grid gap-2">
                        <Label htmlFor="password" className="sr-only">
                           Password
                        </Label>

                        <Input
                           id="password"
                           type="password"
                           name="password"
                           ref={passwordInput}
                           value={data.password}
                           onChange={(e) => setData('password', e.target.value)}
                           placeholder="Password"
                           autoComplete="current-password"
                        />

                        <InputError message={errors.password} />
                     </div>

                     <DialogFooter className="gap-2">
                        <DialogClose asChild>
                           <Button type="button" variant="secondary" onClick={closeModal}>
                              Cancel
                           </Button>
                        </DialogClose>

                        <LoadingButton variant="destructive" loading={processing} className="text-primary-foreground">
                           Delete account
                        </LoadingButton>
                     </DialogFooter>
                  </form>
               </DialogContent>
            </Dialog>
         </div>
      </Card>
   );
}
